package org.codefilarete.stalactite.dsl.entity;

import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.function.Consumer;
import java.util.function.Function;
import javax.annotation.Nullable;

import org.codefilarete.reflection.ReversibleAccessor;
import org.codefilarete.reflection.ValueAccessPoint;
import org.codefilarete.stalactite.dsl.PolymorphismPolicy;
import org.codefilarete.stalactite.dsl.RelationalMappingConfiguration;
import org.codefilarete.stalactite.dsl.embeddable.EmbeddableMappingConfiguration;
import org.codefilarete.stalactite.dsl.idpolicy.IdentifierPolicy;
import org.codefilarete.stalactite.dsl.naming.AssociationTableNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.ColumnNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.ElementCollectionTableNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.ForeignKeyNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.JoinColumnNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.MapEntryTableNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.TableNamingStrategy;
import org.codefilarete.stalactite.dsl.naming.UniqueConstraintNamingStrategy;
import org.codefilarete.stalactite.sql.ddl.Size;
import org.codefilarete.stalactite.sql.ddl.structure.Table;
import org.codefilarete.stalactite.sql.result.ColumnedRow;
import org.codefilarete.tool.collection.ReadOnlyIterator;

/**
 * Defines elements needed to configure a mapping of an entity class
 * 
 * @author Guillaume Mary
 */
public interface EntityMappingConfiguration<C, I> extends RelationalMappingConfiguration<C> {
	
	@Nullable
	EntityFactoryProvider<C, ?> getEntityFactoryProvider();
	
	/**
	 * Gives the table owning entity data.
	 * @return may return null, then a table name will be deduced from entity name and {@link TableNamingStrategy}
	 */
	@Nullable
	Table getTable();
	
	@Nullable
	TableNamingStrategy getTableNamingStrategy();
	
	@Nullable
	ColumnNamingStrategy getColumnNamingStrategy();
	
	KeyMapping<C, I> getKeyMapping();
	
	EmbeddableMappingConfiguration<C> getPropertiesMapping();
	
	@Nullable
	OptimisticLockOption<C, ?> getOptimisticLockOption();
	
	/** Gives inheritance information if inheritance has been defined, else returns null */
	@SuppressWarnings("squid:S1452" /* Can't remove wildcard here because it requires to create a local generic "super" type which is forbidden */)
	@Nullable
	InheritanceConfiguration<? super C, I> getInheritanceConfiguration();
	
	@Nullable
	ForeignKeyNamingStrategy getForeignKeyNamingStrategy();
	
	@Nullable
	UniqueConstraintNamingStrategy getUniqueConstraintNamingStrategy();
	
	@Nullable
	AssociationTableNamingStrategy getAssociationTableNamingStrategy();
	
	@Nullable
	ElementCollectionTableNamingStrategy getElementCollectionTableNamingStrategy();
	
	@Nullable
	JoinColumnNamingStrategy getJoinColumnNamingStrategy();
	
	/**
	 * Gives {@link ColumnNamingStrategy} for index column of one-to-many {@link List} association
	 * @return maybe null, {@link ColumnNamingStrategy#INDEX_DEFAULT} will be used instead
	 */
	@Nullable
	ColumnNamingStrategy getIndexColumnNamingStrategy();
	
	@Nullable
	MapEntryTableNamingStrategy getEntryMapTableNamingStrategy();
	
	PolymorphismPolicy<C> getPolymorphismPolicy();
	
	/**
	 * @return an iterable for all inheritance configurations, including this
	 */
	default Iterable<EntityMappingConfiguration<? super C, I>> inheritanceIterable() {
		
		return () -> new ReadOnlyIterator<EntityMappingConfiguration<? super C, I>>() {
			
			private EntityMappingConfiguration<? super C, I> next = EntityMappingConfiguration.this;
			
			@Override
			public boolean hasNext() {
				return next != null;
			}
			
			@Override
			public EntityMappingConfiguration<? super C, I> next() {
				if (!hasNext()) {
					// comply with next() method contract
					throw new NoSuchElementException();
				}
				EntityMappingConfiguration<? super C, I> result = this.next;
				this.next = org.codefilarete.tool.Nullable.nullable(this.next.getInheritanceConfiguration()).map(InheritanceConfiguration::getConfiguration).get();
				return result;
			}
		};
	}
	
	interface EntityFactoryProvider<C, T extends Table> {

		Function<ColumnedRow, C> giveEntityFactory(T table);
		
		boolean isIdentifierSetByFactory();
	}
	
	interface InheritanceConfiguration<E, I> {
		
		/** Entity configuration */
		EntityMappingConfiguration<E, I> getConfiguration();
		
		boolean isJoinTable();
		
		/** Table to be used in case of joined tables ({@link #isJoinTable()} returns true) */
		@Nullable
		Table getTable();
	}
	
	interface KeyMapping<C, I> {
		
		ReversibleAccessor<C, I> getAccessor();
		
		boolean isSetByConstructor();
	}
	
	interface SingleKeyMapping<C, I> extends KeyMapping<C, I> {
		
		IdentifierPolicy<I> getIdentifierPolicy();
		
		ColumnLinkageOptions getColumnOptions();
		
		@Nullable
		String getFieldName();
	}
	
	interface CompositeKeyMapping<C, I> extends KeyMapping<C, I> {
		
		Consumer<C> getMarkAsPersistedFunction();
		
		Function<C, Boolean> getIsPersistedFunction();
	}
	
	interface ColumnLinkageOptions {
		
		@Nullable
		String getColumnName();
		
		@Nullable
		Size getColumnSize();
		
	}
	
	interface CompositeKeyLinkageOptions {
		
		Map<ValueAccessPoint, String> getColumnsNames();
		
	}
}
